// controllers/collegeComparisonController.js
const db = require("../db");

exports.createComparison = async (req, res) => {
  try {
    const comparisonData = req.body;
    const comparison = await db.CollegeComparison.create(comparisonData);

    res.status(201).json({
      success: true,
      message: "College comparison data created successfully",
      data: comparison,
    });
  } catch (err) {
    console.error("Error:", err);
    res.status(500).json({
      success: false,
      error: "Failed to create college comparison data",
    });
  }
};

exports.getComparisonById = async (req, res) => {
  try {
    const comparison = await db.CollegeComparison.findByPk(req.params.id, {
      include: [
        {
          model: db.College,
          as: "college",
        },
      ],
    });

    if (!comparison) {
      return res.status(404).json({
        success: false,
        error: "Comparison data not found",
      });
    }

    res.json({
      success: true,
      data: comparison,
    });
  } catch (err) {
    console.error("Error:", err);
    res.status(500).json({
      success: false,
      error: "Failed to fetch comparison data",
    });
  }
};

exports.compareColleges = async (req, res) => {
  try {
    const { college1Id, college2Id } = req.params;

    // First verify both colleges exist
    const colleges = await db.College.findAll({
      where: {
        college_id: [college1Id, college2Id],
      },
      attributes: ["college_id", "college_name", "city_name", "logo_image_url"],
    });

    if (colleges.length !== 2) {
      const missingIds = [];
      if (!colleges.some((c) => c.college_id == college1Id))
        missingIds.push(college1Id);
      if (!colleges.some((c) => c.college_id == college2Id))
        missingIds.push(college2Id);

      return res.status(404).json({
        success: false,
        error: "Colleges not found",
        missingIds,
      });
    }

    // Get comparison data
    const comparisons = await db.CollegeComparison.findAll({
      where: {
        fk_college_id: [college1Id, college2Id],
      },
      raw: true,
    });

    // Combine the data
    const result = {
      college1: {
        ...colleges.find((c) => c.college_id == college1Id).toJSON(),
        comparison:
          comparisons.find((c) => c.fk_college_id == college1Id) || {},
      },
      college2: {
        ...colleges.find((c) => c.college_id == college2Id).toJSON(),
        comparison:
          comparisons.find((c) => c.fk_college_id == college2Id) || {},
      },
    };

    res.json({
      success: true,
      data: result,
    });
  } catch (err) {
    console.error("Error comparing colleges:", err);
    res.status(500).json({
      success: false,
      error: "Failed to compare colleges",
      details: process.env.NODE_ENV === "development" ? err.message : undefined,
    });
  }
};

exports.updateComparison = async (req, res) => {
  try {
    const comparison = await db.CollegeComparison.findByPk(req.params.id);
    if (!comparison) {
      return res.status(404).json({
        success: false,
        error: "Comparison data not found",
      });
    }

    await comparison.update(req.body);
    res.json({
      success: true,
      message: "Comparison data updated successfully",
      data: comparison,
    });
  } catch (err) {
    console.error("Error:", err);
    res.status(500).json({
      success: false,
      error: "Failed to update comparison data",
    });
  }
};

exports.deleteComparison = async (req, res) => {
  try {
    const comparison = await db.CollegeComparison.findByPk(req.params.id);
    if (!comparison) {
      return res.status(404).json({
        success: false,
        error: "Comparison data not found",
      });
    }

    await comparison.destroy();
    res.json({
      success: true,
      message: "Comparison data deleted successfully",
    });
  } catch (err) {
    console.error("Error:", err);
    res.status(500).json({
      success: false,
      error: "Failed to delete comparison data",
    });
  }
};
