const db = require("../../db");
const path = require("path");
const fs = require("fs");
const { convertBufferFields } = require("../../utils/bufferUtils");

const StoryController = {
  createStory: async (req, res) => {
    try {
      const story = await db.PPStory.create(req.body);
      res.status(201).json({
        success: true,
        data: convertBufferFields(story),
      });
    } catch (error) {
      res.status(500).json({
        success: false,
        error: error.message,
      });
    }
  },

  getAllStories: async (req, res) => {
    try {
      const { page = 1, limit = 10, search } = req.query;
      const offset = (page - 1) * limit;

      const where = {};
      if (search) {
        where.pps_title = { [db.Sequelize.Op.like]: `%${search}%` };
      }

      const result = await db.PPStory.findAndCountAll({
        where,
        include: [
          {
            model: db.PPStoryData,
            as: "storyData",
            order: [["ppsd_order", "ASC"]],
          },
        ],
        order: [["pps_date_created", "DESC"]],
        limit: parseInt(limit),
        offset: parseInt(offset),
      });

      res.json({
        success: true,
        data: result.rows.map(convertBufferFields),
        pagination: {
          total: result.count,
          pages: Math.ceil(result.count / limit),
          currentPage: parseInt(page),
          perPage: parseInt(limit),
        },
      });
    } catch (error) {
      res.status(500).json({
        success: false,
        error: error.message,
      });
    }
  },

  getStoryWithData: async (req, res) => {
    try {
      const story = await db.PPStory.findByPk(req.params.id, {
        include: [
          {
            model: db.PPStoryData,
            as: "storyData",
            order: [["ppsd_order", "ASC"]],
          },
        ],
      });

      if (!story) {
        return res.status(404).json({
          success: false,
          error: "Story not found",
        });
      }

      res.json({
        success: true,
        data: convertBufferFields(story),
      });
    } catch (error) {
      res.status(500).json({
        success: false,
        error: error.message,
      });
    }
  },

  getStoryImage: async (req, res) => {
    try {
      const filePath = path.join(
        __dirname,
        "../../uploads",
        req.params.filename
      );
      if (fs.existsSync(filePath)) {
        return res.sendFile(filePath);
      }
      res.status(404).send("Image not found");
    } catch (error) {
      res.status(500).json({
        success: false,
        error: error.message,
      });
    }
  },

  addStoryData: async (req, res) => {
    try {
      const story = await db.PPStory.findByPk(req.params.id);
      if (!story) {
        return res.status(404).json({
          success: false,
          error: "Story not found",
        });
      }

      const createdData = await db.PPStoryData.create({
        ...req.body,
        fk_pps_id: req.params.id,
      });

      res.status(201).json({
        success: true,
        data: createdData,
      });
    } catch (error) {
      res.status(500).json({
        success: false,
        error: error.message,
      });
    }
  },

  updateStory: async (req, res) => {
    try {
      const [updated] = await db.PPStory.update(req.body, {
        where: { pps_id: req.params.id },
      });

      if (!updated) {
        return res.status(404).json({
          success: false,
          error: "Story not found",
        });
      }

      const updatedStory = await db.PPStory.findByPk(req.params.id, {
        include: [
          {
            model: db.PPStoryData,
            as: "storyData",
          },
        ],
      });

      res.json({
        success: true,
        data: convertBufferFields(updatedStory),
      });
    } catch (error) {
      res.status(500).json({
        success: false,
        error: error.message,
      });
    }
  },

  deleteStory: async (req, res) => {
    try {
      await db.sequelize.transaction(async (t) => {
        await db.PPStoryData.destroy({
          where: { fk_pps_id: req.params.id },
          transaction: t,
        });

        const deleted = await db.PPStory.destroy({
          where: { pps_id: req.params.id },
          transaction: t,
        });

        if (!deleted) {
          throw new Error("Story not found");
        }
      });

      res.json({
        success: true,
        message: "Story and all associated data deleted successfully",
      });
    } catch (error) {
      const status = error.message === "Story not found" ? 404 : 500;
      res.status(status).json({
        success: false,
        error: error.message,
      });
    }
  },
};

module.exports = StoryController;
